<?php


/**
 * Restituisce una lista dei plugin installati sul wp passato.
 *
 * @param int|null $blog_id
 *
 * @return array
 */
function _getPluginData($blog_id = null) {
    // Prende i plugin presenti:
    $wpPlugins = get_plugins();

    // Switch sul giusto blog:
    if (is_multisite() && $blog_id > 0) {
        switch_to_blog($blog_id);
    }

    // Prende i plugin attivi:
    $activePlugin = (array)get_option('active_plugins');

    // Prende i plugin installati in tutto il network:
    if (is_multisite()) {
        $networkActivePlugin = (array)get_site_option('active_sitewide_plugins');
        $activePlugin = array_merge($activePlugin, $networkActivePlugin);
    }

    // Prende i dati dai plugin installati:
    $pluginData = array();
    if (count($wpPlugins) > 0) {
        foreach ($wpPlugins as $k => $v) {
            $isActive = false;
            if (@in_array($k, $activePlugin)) {
                $isActive = true;
            }

            $pluginData[] = array(
                "id"        => $k,
                "name"      => $v['Name'],
                "version"   => $v['Version'],
                "is_active" => $isActive
            );
        }
    }

    return $pluginData;
}


/**
 * Prende la lista dei blog presenti sul multisite
 */
function _getBlogs() {
    global $wpdb;

    // Variabile di ritorno:
    $blogs = array();

    $blogRis = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT blog_id, domain, path
                FROM $wpdb->blogs
                WHERE
                    site_id = %d AND
                    public = '1' AND
                    archived = '0' AND
                    mature = '0' AND
                    spam = '0' AND
                    deleted = '0'
                ORDER BY registered DESC",
            $wpdb->siteid
        ), ARRAY_A
    );

    if (count($blogRis) > 0) {
        foreach ($blogRis as $row) {
            switch_to_blog($row['blog_id']);

            $blogs[$row['blog_id']] = array(
                "domain"    => site_url(),      // Per ogni blog prende comunque il site url in modo da tenere in considerazione eventuali filtri
                "login_url" => wp_login_url()
            );
        }
    }

    return $blogs;
}


/**
 * Prende sul db gli utenti amministratore.
 *
 * @parm int $blog_id
 *
 * @return array
 */
function _getAdminUser($blog_id = null) {
    $ret = array();

    // Prefisso della tabella del blog:
    if (is_multisite() && $blog_id > 0) {
        switch_to_blog($blog_id);
    }

    /**
     * Esclude i seguenti ruoli, che per definizione non dovrebbero
     * essere amministratori.
     */
    $roleToExclude = array(
        // WP default role
        'subscriber',
        'contributor',
        // BBPress role
        'bbp_moderator',
        'bbp_participant',
        'bbp_spectator',
        'bbp_blocked'
    );

    // Prende gli utenti appartenti al blog:
    $blogUsers = get_users_of_blog();

    if (count($blogUsers) > 0) {
        foreach ($blogUsers as $userData) {
            $user = new WP_User($userData->user_id, null, $blog_id);
            $userRole = $user->roles;

            // Controlla che l'utente possa pubblicare post o pagine:
            if ($user->has_cap('publish_posts') || $user->has_cap('publish_pages')) {
                // Esclude i ruoli non permessi:
                $allowedRoles = array_diff($userRole, $roleToExclude);

                // Verifica che l'utente abbia almeno un ruolo permesso:
                if (count($allowedRoles) > 0) {
                    $ret[] = $userData->user_login;
                }
            }
        }
    }

    return $ret;
}


/**
 * Alcuni plugin di sicurezza possono sovrascrivere la versione di Wordpress.
 * Per ottenere la versione reale usare questa funzione.
 */
function get_real_wp_version() {
    include(ABSPATH . "/wp-includes/version.php");

    return $wp_version;
}


/**
 * Prende il contenuto del wp-config.php, lo modifica per le esigenze del software
 * e lo resituisce
 *
 * @param string $configFilePath
 *
 * @return string
 */
function get_rewritten_wpconfig($configFilePath) {
    // Codice nel file:
    $configPhp = @file_get_contents($configFilePath);

    // Elimina l'apertura di <?php iniziale:
    $configPhp = preg_replace("/^\<\?php/i", "", $configPhp);

    // Aggiunge delle righe vuote alla fine del file:
    $configPhp .= "\n\n";

    /**
     *  Sostituisce la costante __FILE__ con il nome del file config.
     *  Questo per risolvere i problemi che derivano da un stringa nel config tipo:
     *          define( 'WP_CONTENT_DIR', dirname(__FILE__) . '/gears' );
     *
     */
    $configPhp = str_replace("__FILE__", "'" . $configFilePath . "'", $configPhp);

    /**
     * Include lo script di inizializzazione dopo la definizione di:
     *      DB_HOST, DB_USER, DB_PASSWORD, DB_NAME,
     *      MULTISITE (se presente), DOMAIN_CURRENT_SITE (se presente)
     */
    $minPos = 0;
    $defines = array('DB_HOST', 'DB_USER', 'DB_PASSWORD', 'DB_NAME', 'MULTISITE', 'DOMAIN_CURRENT_SITE');
    foreach ($defines as $define) {
        $pos = _findConstantDefineInCode("DB_HOST", $configPhp);
        $minPos = max($minPos, $pos);
    }

    // Lo include dopo il table_prefix:
    $matches = array();
    $found = preg_match("/\\\$table_prefix/i", $configPhp, $matches);
    if ($found) {
        // Posizione nella codice:
        $pos = strpos($configPhp, $matches[0]);
        $minPos = max($minPos, $pos);
    }

    // Lo include prima dell'inclusione di wp-setting.php.
    $maxPos = 0;
    $maxPos = _findFileInclusionInCode("wp-settings.php", $configPhp, "require_once");

    // Lo include prima dell'uso della variabile $_SERVER['HTTP_HOST']
    $matches = array();
    $found = preg_match("/\\\$_SERVER\[[^'\"]*(['\"])HTTP_HOST\g1[^'\"]*\]/i", $configPhp, $matches);
    if ($found) {
        // Posizione nella codice:
        $pos = strpos($configPhp, $matches[0]);
        if ($pos !== false) {
            $maxPos = min($maxPos, $pos);
        }
    }

    // Errore: in tutti casi non è possibile aggiungere lo script necessario.
    // Controllare il wp-config.php e risolvere i conflitti.
    if ($maxPos < $minPos) {
        return false;
    }

    // La prima riga dopo la minima posizione:
    $posNL = strpos($configPhp, "\n", $minPos);
    if ($posNL === false) {
        $posNL = strlen($configPhp);
    }

    // Stringa da includere:
    $filePath = dirname(__FILE__) . "/wp_default_domain_init.php";
    $inclusion = "\ninclude_once(\"" . $filePath . "\");\n";

    // Include lo script nella posizione richiesta:
    $newConfigPhp = substr_replace($configPhp, $inclusion, $posNL, 0);

    return $newConfigPhp;
}


/**
 * Trova la posizione in cui è definita in una stringa una costante.
 *
 * @param string $costantName
 * @param string $phpCode
 * @param string $value
 *
 * @return false|int
 */
function _findConstantDefineInCode($costantName, $phpCode, &$value = null) {
    // Cerca la definizione della costante:
    $matches = array();
    $found = preg_match("/\n[^\n]*define\([^'\"]*(['\"])" . quotemeta($costantName) . "\g1[^'\"\n]*(['\"])([^\g2\n]*)\g2/i", $phpCode, $matches);
    if ($found == 0) {
        return false;
    }

    // Riga:
    $row = str_replace(array("\n", "\r"), "", $matches[0]);

    // Valore della costante:
    $value = $matches[3];

    // Posizione nella riga:
    $pos = strpos($phpCode, $row);

    return $pos;
}


/**
 * Trova la posizione in cui è definita l'inclusione di un file.
 *
 * @param string $filename
 * @param string $phpCode
 * @param string $statement
 *
 * @return false|int
 */
function _findFileInclusionInCode($filename, $phpCode, $statement = "require_once") {
    // Cerca l'inclusione:
    $matches = array();
    $found = preg_match("/\n[^\n]*" . quotemeta($statement) . "[^'\"]*(['\"])" . quotemeta($filename) . "\g1[^\n]*\n/i", $phpCode, $matches);
    if ($found == 0) {
        return false;
    }

    // Riga:
    $row = str_replace(array("\n", "\r"), "", $matches[0]);

    // Posizione nella riga:
    $pos = strpos($phpCode, $row);

    return $pos;
}

