<?php

/**
 * Restituisce la quota usata e disponibile relativa ad un account
 *
 * @param string $userEmail     Acconut
 * @param string $clientEmail   Superadmin
 * @param string $privateKey
 * @param array $errorMsg       Eventuali messaggi di errore
 *
 * @return boolean
 */
function getUserAccountQuota($userEmail, $clientEmail, $privateKey, &$errorMsg = array()) {
    // Scopes per le chiamate da effettuare:
    $scopes = array(
        Google_Service_Drive::DRIVE_METADATA_READONLY
    );

    // Credenziali di accesso:
    $credentials = new Google_Auth_AssertionCredentials(
        $clientEmail,
        $scopes,
        $privateKey,
        'notasecret',
        'http://oauth.net/grant_type/jwt/1.0/bearer',
        $userEmail
    );

    // Client:
    try {
        $client = new Google_Client();
        $client->setAssertionCredentials($credentials);
        if ($client->getAuth()->isAccessTokenExpired()) {
            $client->getAuth()->refreshTokenWithAssertion();
        }

        // API:
        $driveApi = new Google_Service_Drive($client);

        // Dati di Drive:
        $driveData = $driveApi->about->get(array(
            'fields' => 'quotaBytesByService,quotaBytesTotal,quotaBytesUsed,quotaBytesUsedAggregate,quotaType'
        ));

        $quotaTotale = $quotaUsata = 0;

        if (!empty($driveData)) {
            // Quota a disposizione dell'utente:
            $quotaTotale = $driveData->quotaBytesTotal;

            // Quota usata in tutte le applicazioni:
            $quotaUsata = $driveData->quotaBytesUsedAggregate;
        }
    } catch (Google_Auth_Exception $ex) {
        // ERRORE: Occorre autorizzare l'app ad accedere alle informazioni del dominio.
        list($user, $domain) = explode("@", $userEmail);
        $errorMsg[] = sprintf("DOMAIN '%s': autorizzazioni API necessarie.", $domain);

        return false;
    } catch (Exception $ex) {
        return false;
    }

    return array(
        "usata" => $quotaUsata,
        "totale" => $quotaTotale
    );
}


/**
 * Restituisce alcune informazioni sui setting di un account
 *
 * @param string $userEmail     Acconut
 * @param string $clientEmail   Superadmin
 * @param string $privateKey
 * @param array $errorMsg       Eventuali messaggi di errore
 *
 * @return boolean
 */
function getUserAccountSettings($userEmail, $clientEmail, $privateKey, &$errorMsg = array()) {
    // Scopes:
    $scopes = array(
        Dbtech_Neikos_Google_Service_EmailSettings::EMAILSETTINGS
    );

    // Credenziali di accesso:
    $credentials = new Google_Auth_AssertionCredentials(
        $clientEmail,
        $scopes,
        $privateKey,
        'notasecret',
        'http://oauth.net/grant_type/jwt/1.0/bearer',
        $userEmail
    );

    // Client:
    try {
        $client = new Dbtech_Neikos_Google_Client();
        $client->setAssertionCredentials($credentials);
        if ($client->getAuth()->isAccessTokenExpired()) {
            $client->getAuth()->refreshTokenWithAssertion();
        }

        // Utente e dominio:
        list($user, $domain) = explode("@", $userEmail);

        // API:
        $emailSettingApi = new Dbtech_Neikos_Google_Service_EmailSettings($client);

        // Default:
        $forward = "";
        $autoresponder = false;

        // Forwarding:
        $response = $emailSettingApi->forwarding->get($domain, $user);
        if ($response->enable) {
            $forward = $response->forwardTo;
        }

        // Autorisponditore:
        $response = $emailSettingApi->vacation->get($domain, $user);
        if ($response->enable) {
            $autoresponder = true;
        }
    } catch (Google_Auth_Exception $ex) {
        // ERRORE: Occorre autorizzare l'app ad accedere alle informazioni del dominio.
        list($user, $domain) = explode("@", $userEmail);
        $errorMsg[] = sprintf("DOMAIN '%s': autorizzazioni API necessarie.", $domain);

        return false;
    } catch (Exception $ex) {
        return false;
    }

    return array(
        "forward"       => $forward,
        "autoresponder" => $autoresponder
    );
}


/**
 * Invia un'email con gli eventuali messaggi di errore
 *
 * @param string $to
 * @param array $errorMsg
 * @param string $clientId
 */
function sendErrorEmail($to, $errorMsg, $clientId) {
    // Oggetto:
    $subject = "DBTECH - Errore accesso API posta da Google Reseller";

    // Messaggio:
    $message = "Gentile amministratore,\n" .
               "durante la scansione periodica dell'account reseller di Neikos su Google, sono stati riscontrati i seguenti errori:\n\n";
    foreach ($errorMsg as $e) {
        $message .= "\t- " . $e . "\n";
    }
    $message .= "\nRicorda che, al fine di riportare i dati sul dbtech, è stata creata un'app su Google Console.\n" .
                "Per ogni nuovo dominio abilitato occorre andare sulla dashboard del dominio e poi in:\n" .
                "\tSicurezza > Riferimento API > Accesso API\n" .
                "e spuntare \"Attiva accesso API\".\n\n" .
                "Inoltre in:\n" .
                "\tSicurezza > Impostazioni avanzate > Gestisci accesso client API\n" .
                "occorre specificare per il clientId dell'app (" . $clientId . ") le seguenti autorizzazioni separate da virgola:\n" .
                "\thttps://www.googleapis.com/auth/drive.metadata.readonly\n" .
                "\thttps://apps-apis.google.com/a/feeds/emailsettings/2.0/\n\n";
    $message .= "Se il problema persiste, occorre indagare su eventuali malfunzionamenti dell'app.\n\n" .
                "Grazie\nBuon lavoro\n\n--\n\n" .
                "Email generata automaticamente, non rispondere\n";

    // Header aggiuntivi:
    $additional_headers = "From: " . $to . "\r\n";

    mail($to, $subject, $message, $additional_headers, "-f " . $to);
}
