<?php

class Database extends Zend_Db_Table_Abstract {
	const IMMESSO_AUTOMATICO = "A";
	const IMMESSO_MANUALE = "M";
	const IMMESSO_AUTOMATICO_ANCHE_CLIENTE = "P";

	protected $_name = 'database';
	protected $_dependentTables = array("DatabaseToDatabaseUser");

	protected $_referenceMap    = array(
        'Servers' => array(
            'columns'           => 'server_id',
            'refTableClass'     => 'Servers',
            'refColumns'        => 'id'
        ),

        'Clients' => array(
            'columns'           => 'project_id',
            'refTableClass'     => 'Clients',
            'refColumns'        => 'ID'
        ),

        'Software' => array(
            'columns'           => 'software_id',
            'refTableClass'     => 'Software',
            'refColumns'        => 'id'
        )
    );


	/**
	 * Prende tutti i dati che saranno visualizzati
	 * in una pagina.
	 *
	 * @param int $numPage - pagina corrente (default: 1).
	 * @param int $maxPerPage - numero massimo di record per pagina (default = 25).
	 * @return mixed
	 */
	public function getPage($numPage = 1, $maxPerPage = 25) {
		$select = $this->select()->from($this)
				       ->limitPage($numPage, $maxPerPage)->order("nome");

		// Applica i filtri di ricerca:
		foreach ($this->_filtriRicerca as $f) {
			$select->where($f);
		}

		return $this->fetchAll($select);
	}

	/**
	 * Conta in numero di record.
	 *
	 * @return int
	 */
	public function count() {
		$select = $this->select();
		$select->from($this, array("num" => "COUNT(id)"));

		// Applica i filtri di ricerca:
		foreach ($this->_filtriRicerca as $f) {
			$select->where($f);
		}

		$rows = $this->fetchAll($select);

		if (count($rows)>0) {
			return (int)$rows->current()->num;
		}
		return 0;
	}


	// Filtri di ricerca:
	protected $_filtriRicerca = null;


	/**
	 * Setta i filtri per la lista dei domini.
	 */
	public function setFiltri($filtriRicerca) {
		$f = array();

		if ($filtriRicerca->client_id != 0) {
			if ($filtriRicerca->client_id == -1) {
				$f[] = "project_id IS NULL OR project_id = 0";
			} else if ($filtriRicerca->client_id == -2) {
				$clients = new Clients();
				$f[] = "project_id IN ('" . implode("', '", $clients->getIdFermati()) . "')";
			} else {
				$f[] = $this->_db->quoteInto("project_id = ?", $filtriRicerca->client_id);
			}
		}
		if ($filtriRicerca->nome != "") {
			$f[] = $this->_db->quoteInto("nome LIKE ?", "%" . $filtriRicerca->nome . "%");
		}
		if ($filtriRicerca->server_id != 0) {
			$f[] = $this->_db->quoteInto("server_id = ?", $filtriRicerca->server_id);
		}
		if (!empty($filtriRicerca->immesso)) {
			$f[] = $this->_db->quoteInto("immesso = ?", $filtriRicerca->immesso);
		}

		$this->_filtriRicerca = $f;
	}


	public static function getLista($params = array()) {
		$model = new self();
		$sel = $model->select()->order("nome ASC");

		if (!empty($params["immesso"])) {
			$sel->where("immesso = ?", $params["immesso"]);
		}

		if ((int)$params["server_id"] > 0) {
			$sel->where("server_id = ?", (int)$params["server_id"]);
		}

		return $model->fetchAll($sel);
	}

	public static function getListaDbForSelect($params = array()) {
		$ris = self::getLista($params);
		$ret = array();

		if (count($ris) > 0) {
			foreach ($ris as $row) {
				// Server associato:
				$server = $row->findParentRow("Servers");
				$serverName = "";

				if (count($server) > 0) {
					$serverName = $server->name;
				}

				$ret[$row->id] = $row->nome . (($serverName != "") ? " (" . $serverName . ")" : "");
			}
		}

		return $ret;
	}


	public static function getListaDbManualiForSelect($server_id = null) {
		$ris = self::getLista(
				array(
					"server_id" => $server_id,
					"immesso" 	=> self::IMMESSO_MANUALE
				)
		);
		$ret = array();

		if (count($ris) > 0) {
			foreach ($ris as $row) {
				// Server associato:
				$server = $row->findParentRow("Servers");
				$serverName = "";

				if (count($server) > 0) {
					$serverName = $server->name;
				}

				$ret[$row->id] = $row->nome . (($serverName != "") ? " (" . $serverName . ")" : "");
			}
		}

		return $ret;
	}


	/**
	 * Lista dei server con utenti database associati.
	 */
	public function listaServerForSelect() {
		$where = "EXISTS (SELECT * FROM " .  $this->getAdapter()->quoteIdentifier($this->_name) . " AS m WHERE m.server_id = s.id)";
		$ris = Servers::getListaPerSelect($where);

		return $ris;
	}



	public function insert(array $data)
    {
		try {
			// Setta le date di inserimento e modifica e l'immissione manuale:
			$data["last_update"] = date("Y-m-d H:i:s");
			$data["immesso"] = self::IMMESSO_MANUALE;

			$id = parent::insert($data);
		} catch	(Exception $ex) {
			// Salva un msg di errore nello stack dei msg:
			Neikos_Systemmessage::push("E' occorso un errore nel salvataggio del database.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);
			return false;
		}

		// Salva un messaggio nel log:
		Neikos_Log::Write(sprintf("Nuovo database: (%s)", $id));

		// Salva un msg di OK nello stack dei msg positivi:
		Neikos_Systemmessage::push("Il database &egrave; stato salvato con successo.");

		return $id;
    }

    public function update(array $data, $where)
    {
    	try {
    		// Setta la data di modifica:
			$data["last_update"] = date("Y-m-d H:i:s");

    		$oldDati = $this->fetchAll($where);

			if (count($oldDati) > 0) {
				$stringa = array();
				foreach ($oldDati as $oldData) {
					$stringa[] = sprintf("Modifica database: (%s), %s, %s, %s, %s, '%s'", $oldData->id, $oldData->nome, $oldData->project_id, $oldData->server_id, $oldData->software_id, base64_encode($oldData->note));
				}
    		}
    	} catch (Exception $ex) {
    		// Nessuna eccezione mostrata...
    	}

		try {
			$numRows = parent::update($data, $where);
		} catch(Exception $ex) {
			// Salva un msg di errore nello stack dei msg:
			Neikos_Systemmessage::push("Errore nella modifica dei record.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);

			return false;
		}

		// Scrive sul file di log:
		Neikos_Log::Write($stringa);

		return $numRows;
    }

    public function delete($where)
    {
        try {
    		$oldDati = $this->fetchAll($where);

			if (count($oldDati) > 0) {
    			$stringa = array();
				foreach ($oldDati as $oldData) {
    				$stringa[] = sprintf("Eliminazione database: (%s), %s, %s, %s, %s, '%s'", $oldData->id, $oldData->nome, $oldData->project_id, $oldData->server_id, $oldData->software_id, base64_encode($oldData->note));
				}
    		}
    	} catch (Exception $ex) {
    		// Nessuna eccezione mostrata...
    	}

		try {
			$numRows = parent::delete($where);
		} catch(Exception $ex) {
			// Salva un msg di errore nello stack dei msg:
			Neikos_Systemmessage::push("Errore nell'eliminazione dei record.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);

			return false;
		}

		// Scrive nel log:
		Neikos_Log::Write($stringa);

		// Salva un msg di OK nello stack dei msg positivi:
		Neikos_Systemmessage::push(sprintf("L'eliminazione &egrave; stata efettuata con successo. Record eliminati: %d", $numRows));

		return $numRows;
    }
}

?>