<?php

class DatabaseUser extends Zend_Db_Table_Abstract {
	protected $_name = 'database_user';
	protected $_dependentTables = array("DatabaseToDatabaseUser");

	protected $_referenceMap    = array(
        'Server' => array(
            'columns'           => 'server_id',
            'refTableClass'     => 'Servers',
            'refColumns'        => 'id'
        )
    );

    const IMMESSO_AUTOMATICO = "A";
    const IMMESSO_MANUALE = "M";
    const IMMESSO_AUTOMATICO_ANCHE_PASSSWORD = "P";


	// Filtri di ricerca:
	protected $_filtriRicerca = null;


	/**
	 * Prende tutti i dati che saranno visualizzati
	 * in una pagina.
	 *
	 * @param int $numPage - pagina corrente (default: 1).
	 * @param int $maxPerPage - numero massimo di record per pagina (default = 25).
	 * @return mixed
	 */
	public function getPage($numPage = 1, $maxPerPage = 25) {
		$select = $this->select()->from(array("du" => $this->_name))
				 ->limitPage($numPage, $maxPerPage)->order("du.server_id")->order("du.user");

		// Applica i filtri di ricerca:
		foreach ($this->_filtriRicerca as $f) {
			$select->where($f);
		}

		return $this->fetchAll($select);
	}

	/**
	 * Conta in numero di record.
	 *
	 * @return int
	 */
	public function count() {
		$select = $this->select();
		$select->from(array("du" => $this->_name), array("num" => "COUNT(id)"));

		// Applica i filtri di ricerca:
		foreach ($this->_filtriRicerca as $f) {
			$select->where($f);
		}

		$rows = $this->fetchAll($select);

		if (count($rows) > 0) {
			return (int)$rows->current()->num;
		}
		return 0;
	}

	/**
	 * Setta i filtri per la lista dei domini.
	 */
	public function setFiltri($filtriRicerca) {
		$f = array();

		if ($filtriRicerca->user != "") {
			$f[] = $this->_db->quoteInto("du.user LIKE ?", "%" . $filtriRicerca->user . "%");
		}
		if ($filtriRicerca->server_id != 0) {
			$f[] = $this->_db->quoteInto("du.server_id = ?", $filtriRicerca->server_id);
		}
		if ($filtriRicerca->database_id != 0) {
			$f[] = $this->_db->quoteInto("EXISTS (SELECT * FROM database_to_database_user AS dtdu WHERE dtdu.database_user_id=du.id AND dtdu.database_id = ?)", $filtriRicerca->database_id);
		}
		if ($filtriRicerca->senza_pw == "A") {
			$f[] = "du.pw = ''";
		}
		if (!empty($filtriRicerca->immesso)) {
			$f[] = $this->_db->quoteInto("du.immesso = ?", $filtriRicerca->immesso);
		}

		$this->_filtriRicerca = $f;
	}

	/**
	 * Lista dei server con utenti database associati.
	 */
	public function listaServerForSelect() {
		$where = "EXISTS (SELECT * FROM " . $this->getAdapter()->quoteIdentifier($this->_name) ." AS m WHERE m.server_id = s.id)";
		$ris = Servers::getListaPerSelect($where);

		return $ris;
	}


	public function insert(array $data)
    {
		try {
			// Setta le date di inserimento e modifica,
			// l'immissione manuale e il server:
			$data["update_status"] = "A";
			$data["last_update"] = date("Y-m-d H:i:s");
			$data["immesso"] = self::IMMESSO_MANUALE;

			$id = parent::insert($data);
		} catch	(Exception $ex) {
			// Salva un msg di errore nello stack dei msg:
			Neikos_Systemmessage::push("E' occorso un errore nel salvataggio dell'utente database.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);
			return false;
		}

		// Salva un messaggio nel log:
		Neikos_Log::Write(sprintf("Nuovo utente database: (%s)", $id));

		// Salva un msg di OK nello stack dei msg positivi:
		Neikos_Systemmessage::push("L'utente database &egrave; stato salvato con successo.");

		return $id;
    }

    public function update(array $data, $where)
    {
    	try {
    		// Setta la data di modifica:
			$data["last_update"] = date("Y-m-d H:i:s");

    		$oldDati = $this->fetchAll($where);

			if (count($oldDati) > 0) {
				$stringa = array();
				foreach ($oldDati as $oldData) {
					$stringa[] = sprintf("Modifica utente database: (%s), %s, %s, %s, %s, %s, '%s'", $oldData->id, $oldData->host, $oldData->user, $oldData->password, $oldData->server_id, base64_encode($oldData->privilegi), base64_encode($oldData->note));
				}
    		}
    	} catch (Exception $ex) {
    		// Nessuna eccezione mostrata...
    	}

		try {
			$numRows = parent::update($data, $where);
		} catch(Exception $ex) {
			// Salva un msg di errore nello stack dei msg:
			Neikos_Systemmessage::push("Errore nella modifica dei record.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);

			return false;
		}

		// Scrive sul file di log:
		Neikos_Log::Write($stringa);

		return $numRows;
    }

    public function delete($where)
    {
        try {
    		$oldDati = $this->fetchAll($where);

			if (count($oldDati) > 0) {
    			$stringa = array();
				foreach ($oldDati as $oldData) {
    				$stringa[] = sprintf("Eliminazione utente database: (%s), %s, %s, %s, %s, %s, '%s'", $oldData->id, $oldData->host, $oldData->user, $oldData->password, $oldData->server_id, base64_encode($oldData->privilegi), base64_encode($oldData->note));
				}
    		}
    	} catch (Exception $ex) {
    		// Nessuna eccezione mostrata...
    	}

		try {
			$numRows = parent::delete($where);
		} catch(Exception $ex) {
			// Salva un msg di errore nello stack dei msg:
			Neikos_Systemmessage::push("Errore nell'eliminazione dei record.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);

			return false;
		}

		// Scrive nel log:
		Neikos_Log::Write($stringa);

		// Salva un msg di OK nello stack dei msg positivi:
		Neikos_Systemmessage::push(sprintf("L'eliminazione &egrave; stata efettuata con successo. Record eliminati: %d", $numRows));

		return $numRows;
    }
}

