<?php

class Dns extends Zend_Db_Table_Abstract {
    const IMMESSO_AUTOMATICO = "A";
    const IMMESSO_MANUALE = "M";

	protected $_name = 'dns';
	protected $_dependentTables = array();

	protected $_referenceMap    = array(
        'Server' => array(
            'columns'           => 'server_id',
            'refTableClass'     => 'Servers',
            'refColumns'        => 'id'
        ),

        'Clients' => array(
            'columns'           => 'project_id',
            'refTableClass'     => 'Clients',
            'refColumns'        => 'ID'
        )
    );

	/**
	 * Prende tutti i dati che saranno visualizzati
	 * in una pagina.
	 *
	 * @param int $numPage - pagina corrente (default: 1).
	 * @param int $maxPerPage - numero massimo di record per pagina (default = 25).
	 * @return mixed
	 */
	public function getPage($numPage = 1, $maxPerPage = 25) {
		$select = $this->select()->from($this, array("*"))
				 ->order("name");

		if ($numPage >= 0 && $maxPerPage !== null) {
			$select->limitPage($numPage, $maxPerPage);
		}

		// Applica i filtri di ricerca:
		foreach ($this->_filtriRicerca as $f) {
			$select->where($f);
		}

		return $this->fetchAll($select);
	}

	/**
	 * Conta in numero di record.
	 *
	 * @return int
	 */
	public function count() {
		$select = $this->select();
		$select->from($this, array("num" => "COUNT(id)"));

		// Applica i filtri di ricerca:
		foreach ($this->_filtriRicerca as $f) {
			$select->where($f);
		}

		$rows = $this->fetchAll($select);

		if (count($rows)>0) {
			return (int)$rows->current()->num;
		}
		return 0;
	}

	// Filtri di ricerca:
	protected $_filtriRicerca = null;

	/**
	 * Setta i filtri per la lista dei domini.
	 */
	public function setFiltri($filtriRicerca) {
		$f = array();

		if ($filtriRicerca->client_id != 0) {
			if ($filtriRicerca->client_id == -1) {
				$f[] = "project_id IS NULL OR project_id = 0";
			} else if ($filtriRicerca->client_id == -2) {
				$clients = new Clients();
				$f[] = "project_id IN ('" . implode("', '", $clients->getIdFermati()) . "')";
			} else {
				$f[] = $this->_db->quoteInto("project_id = ?", $filtriRicerca->client_id);
			}
		}
		if ($filtriRicerca->server_id > 0) {
			$f[] = $this->_db->quoteInto("server_id = ?", $filtriRicerca->server_id);
		}
		if ($filtriRicerca->name != "") {
			$f[] = $this->_db->quoteInto("name LIKE ?", "%" . $filtriRicerca->name . "%");
		}
		if (!empty($filtriRicerca->immesso)) {
		    $f[] = $this->_db->quoteInto("immesso = ?", $filtriRicerca->immesso);
		}

		$this->_filtriRicerca = $f;
	}

    public function insert(array $data)
    {
		try {
			$id = parent::insert($data);
		} catch	(Exception $ex) {
			// Salva un msg di errore nello stack dei msg:
			Neikos_Systemmessage::push("E' occorso un errore nel salvataggio del dns.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);
			return false;
		}

		// Salva un messaggio nel log:
		Neikos_Log::Write(sprintf("Nuovo dns: (%s)", $id));

		// Salva un msg di OK nello stack dei msg positivi:
		Neikos_Systemmessage::push("Il dns &egrave; stato salvato con successo.");

		return $id;
    }

	public function update(array $data, $where) {
		try {
    		$oldDati = $this->fetchAll($where);

			if (count($oldDati) > 0) {
				$stringa = array();
				foreach ($oldDati as $oldData) {
					$stringa[] = sprintf("Modifica dns: (%d), %s, %s, %s", $oldData->id, $oldData->name, $oldData->project_id, $oldData->server_id);
				}
    		}
    	} catch (Exception $ex) {
    		// Nessuna eccezione mostrata...
    	}

		try {
			$numRows = parent::update($data, $where);
		} catch(Exception $ex) {
			// Salva un msg di errore nello stack dei msg:
			Neikos_Systemmessage::push("Errore nella modifica dei record.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);

			return false;
		}

		// Scrive sul file di log:
		Neikos_Log::Write($stringa);

    	return $numRows;
	}

	public function delete($where) {
		try {
    		$oldDati = $this->fetchAll($where);

			if (count($oldDati) > 0) {
    			$stringa = array();
				foreach ($oldDati as $oldData) {
    				$stringa[] = sprintf("Eliminazione dns: (%s), %s, %s, %s", $oldData->id, $oldData->name, $oldData->project_id, $oldData->server_id);
				}
    		}
    	} catch (Exception $ex) {
    		// Nessuna eccezione mostrata...
    	}

		try {
			$numRows = parent::delete($where);
		} catch(Exception $ex) {
			// Salva un msg di errore nello stack dei msg:
			Neikos_Systemmessage::push("Errore nell'eliminazione dei record.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);

			return false;
		}

		// Scrive nel log:
		Neikos_Log::Write($stringa);

		// Salva un msg di OK nello stack dei msg positivi:
		Neikos_Systemmessage::push(sprintf("L'eliminazione &egrave; stata efettuata con successo. Record eliminati: %d", $numRows));

		return $numRows;
	}
}
