<?php

class Software extends Zend_Db_Table_Abstract {
	protected $_name = 'software';
	protected $_dependentTables = array('Database');

	const USER_TIPO_SUPERADMIN = "S";
	const USER_TIPO_ADMIN = "A";
	const USER_TIPO_NORMAL = "N";

	const URL_TIPO_SUPERADMIN = "S";
	const URL_TIPO_ADMIN = "A";
	const URL_TIPO_PUBBLICA = "P";

	const IMMESSO_AUTOMATICO = "A";
	const IMMESSO_MANUALE = "M";

	// Classe custom per la riga sul db:
	protected $_rowClass = "Software_Row";

	public static function getTipiUser() {
		return array(
			self::USER_TIPO_SUPERADMIN 	=> "super admin",
			self::USER_TIPO_ADMIN 		=> "admin",
			self::USER_TIPO_NORMAL 		=> "normal user"
		);
	}

	public static function getTipiUrl() {
		return array(
			self::URL_TIPO_SUPERADMIN	=> "super admin",
			self::URL_TIPO_ADMIN 		=> "admin",
			self::URL_TIPO_PUBBLICA		=> "pubblica"
		);
	}

	protected $_referenceMap    = array(
	    'Server' => array(
	        'columns'           => 'server_id',
	        'refTableClass'     => 'Servers',
	        'refColumns'        => 'id'
	    ),

		'Clients' => array(
            'columns'           => 'project_id',
            'refTableClass'     => 'Clients',
            'refColumns'        => 'ID'
        ),

        'Software_Tipo' => array(
            'columns'           => 'software_tipo_id',
            'refTableClass'     => 'Software_Tipo',
            'refColumns'        => 'id'
        )
	);

	public static function findByUrl($url) {
		$model = new self();

		$where = $model->getAdapter()->quoteInto('url LIKE ?', "%" . $url . "%");
        return $model->fetchAll($where);
	}

	public static function getNumByTipo($tipo) {
		$model = new self();

		$sel = $model->select();
		$sel->from($model, array("tot" => "COUNT(*)"));
		$sel->where("software_tipo_id = ?", (int)$tipo);
		$ris = $model->fetchAll($sel);

		if (count($ris)>0) {
			return (int)$ris[0]["tot"];
		}

		return 0;
	}


	/**
	 * Lista software.
	 *
	 * @param int $tipo_id
	 * @return Zend_Db_Table_Rowset
	 */
	public static function getAll() {
	    $model = new self();

	    return $model->fetchAll();
	}

	/**
	 * Lista software per un select.
	 *
	 * @param int $tipo_id
	 * @return Zend_Db_Table_Rowset
	 */
	public static function getAllForSelect() {
	    $rowset = self::getAll();
	    $ret = array();

	    if (count($rowset) > 0) {
	        foreach ($rowset as $row) {
	            $ret[$row->id] = $row->getDescrizione();
	        }
	    }

	    return $ret;
	}


	/**
	 * Lista software per tipo.
	 *
	 * @param int $tipo_id
	 * @return Zend_Db_Table_Rowset
	 */
	public static function getSoftwareByTipo($tipo_id) {
		$model = new self();

		$sel = $model->select();
		$sel->where("software_tipo_id = ?", (int)$tipo_id);

		return $model->fetchAll($sel);
	}

	/**
	 * Lista software per cliente.
	 *
	 * @param int $tipo_id
	 * @return Zend_Db_Table_Rowset
	 */
	public static function getSoftwareByProjectId($project_id) {
		$model = new self();

		$sel = $model->select();
		$sel->where("project_id  = ?", (int)$project_id);

		return $model->fetchAll($sel);
	}

	/**
	 * Lista software per cliente formattati per un select.
	 *
	 * @param int $tipo_id
	 * @return Zend_Db_Table_Rowset
	 */
	public static function getSoftwareByProjectIdForSelect($project_id) {
		$rowset = self::getSoftwareByProjectId($project_id);
		$ret = array();

		if (count($rowset) > 0) {
			foreach ($rowset as $row) {
				$ret[$row->id] = $row->getDescrizione();
			}
		}

		return $ret;
	}


	public function moveByTipo($fromTipo, $toTipo) {
		return $this->update(
			array("software_tipo_id" => $toTipo),
			$this->getAdapter()->quoteInto("software_tipo_id = ?", $fromTipo)
		);
	}

	public function deleteByTipo($tipo) {
		return $this->delete(
			$this->getAdapter()->quoteInto("software_tipo_id = ?", $tipo)
		);
	}

	// Filtri di ricerca:
	protected $_filtriRicerca = null;

	/**
	 * Setta i filtri per la lista dei domini.
	 */
	public function setFiltri($filtriRicerca) {
		$f = array();

		// Alcuni parametri:
		$version = str_replace("x", "%", $filtriRicerca->versione);
		$pluginVersion = str_replace("x", "%", $filtriRicerca->plugin_version);

		if (!empty($filtriRicerca->name)) {
			$f[] = $this->_db->quoteInto("p.note LIKE ?", "%" . $filtriRicerca->name . "%") .
				" OR " .
				$this->_db->quoteInto("p.url LIKE ?", "%" . $filtriRicerca->name . "%") .
				" OR " .
				$this->_db->quoteInto("p.info LIKE ?", "%" . $filtriRicerca->name . "%");
		}

		if (!empty($filtriRicerca->versione)) {
		    $f[] = $this->_db->quoteInto("p.versione LIKE ?", $version);
		}

		if ($filtriRicerca->client_id != 0) {
			if ($filtriRicerca->client_id == -1) {
				$f[] = "p.project_id IS NULL OR p.project_id = 0";
			} else {
				$f[] = $this->_db->quoteInto("p.project_id = ?", $filtriRicerca->client_id);
			}
		}

		if ($filtriRicerca->server_id != 0) {
		    if ($filtriRicerca->server_id == -1) {
		        $f[] = "p.server_id IS NULL OR p.server_id = 0";
		    } else {
		        $f[] = $this->_db->quoteInto("p.server_id = ?", $filtriRicerca->server_id);
		    }
		}

		if ($filtriRicerca->mostra_fermati != 1) {
			$clients = new Clients();
			$f[] = "p.project_id NOT IN ('" . implode("', '", $clients->getIdFermati()) . "')";
		}

		if ($filtriRicerca->software_tipo_id > 0) {
			$f[] = $this->_db->quoteInto("p.software_tipo_id = ?", $filtriRicerca->software_tipo_id);
		}

		if (!empty($filtriRicerca->immesso)) {
		    $f[] = $this->_db->quoteInto("immesso = ?", $filtriRicerca->immesso);
		}

		// PLUGIN:

		// Stato plugin:
		$pluginStatusQuery = "";
		if (!empty($filtriRicerca->plugin_status)) {
		    $pluginStatusQuery = $this->_db->quoteInto(" AND stp.status = ?", $filtriRicerca->plugin_status);
		}

		// Ricerca per plugin e versione:
		if ($filtriRicerca->plugin_ids > 0) {
		    $pluginIds = "'" . implode("', '", $filtriRicerca->plugin_ids)  . "'";

		    if (!empty($filtriRicerca->plugin_version)) {
		        // Ricerca per versione e tipo plugin:
		        $f[] = $this->_db->quoteInto(
		            "EXISTS (SELECT * FROM software_to_plugin AS stp
		                      WHERE p.id = stp.software_id AND stp.software_plugin_id IN (" . $pluginIds . ") AND stp.versione LIKE ?" . $pluginStatusQuery . ")", $pluginVersion
		        );
		    } else {
		        // Ricerca solo per tipo plugin:
		        $f[] = "EXISTS (SELECT * FROM software_to_plugin AS stp WHERE p.id = stp.software_id AND stp.software_plugin_id IN (" . $pluginIds . ")" . $pluginStatusQuery . ")";
		    }
		} else if ($filtriRicerca->plugin_version > 0) {
		    // Ricerca solo per versione:
		    $f[] = $this->_db->quoteInto(
		        "EXISTS (SELECT * FROM software_to_plugin AS stp WHERE p.id = stp.software_id AND stp.versione LIKE ?" . $pluginStatusQuery . ")", $pluginVersion
		    );
		} elseif (!empty($filtriRicerca->plugin_status)) {
		    // Ricerca solo per stato plugin:
		    $f[] = "EXISTS (SELECT * FROM software_to_plugin AS stp WHERE p.id = stp.software_id" . $pluginStatusQuery . ")";
		}

		$this->_filtriRicerca = $f;
	}


	/**
	 * Prende tutti i dati che saranno visualizzati
	 * in una pagina.
	 *
	 * @param int $numPage - pagina corrente (default: 1).
	 * @param int $maxPerPage - numero massimo di record per pagina (default = 25).
	 * @return mixed
	 */
	public function getPage($numPage = 1, $maxPerPage = 25) {
		$adapter = $this->getAdapter();

		$clients = new Clients();
		$cInfo = $clients->info();
		$cName = $cInfo[Zend_Db_Table_Abstract::SCHEMA] . "." . $cInfo[Zend_Db_Table_Abstract::NAME];

		$select = $this->select()
				->from(array("p" => $this->_name))
		     	->setIntegrityCheck(false)
			   	->joinLeft(array("q" => $cName), "p.project_id = q.ID", array())
			   	->order(array("q.name ASC", "p.user ASC"));

		if ($numPage >= 0 && $maxPerPage !== null) {
			$select->limitPage($numPage, $maxPerPage);
		}

		// Applica i filtri di ricerca:
		if (count($this->_filtriRicerca) > 0) {
			foreach ($this->_filtriRicerca as $f) {
				$select->where($f);
			}
		}

		return $this->fetchAll($select);
	}

	/**
	 * Conta in numero di record.
	 *
	 * @return int
	 */
	public function count() {
		$adapter = $this->getAdapter();

		$clients = new Clients();
		$cInfo = $clients->info();
		$cName = $cInfo[Zend_Db_Table_Abstract::SCHEMA] . "." . $cInfo[Zend_Db_Table_Abstract::NAME];

		$select = $this->select()
				->from(array("p" => $this->_name), array("num" => "COUNT(p.id)"))
		     	->setIntegrityCheck(false)
			   	->joinLeft(array("q" => $cName), "p.project_id = q.ID", array())
			   	->order(array("q.name ASC", "p.user ASC"));

		// Applica i filtri di ricerca:
		if (count($this->_filtriRicerca) > 0) {
			foreach ($this->_filtriRicerca as $f) {
				$select->where($f);
			}
		}

		$rows = $this->fetchAll($select);

		if (count($rows)>0) {
			return (int)$rows->current()->num;
		}
		return 0;
	}


	public function insert(array $data)	{
		try {
			// Setta le date di inserimento e modifica:
			$data["data_ins"] = $data["data_mod"] = date("Y-m-d H:i:s");

			$id = parent::insert($data);
		} catch (Exception $ex) {
			// Segna l'errore nello stack dei msg di errore:
			Neikos_Systemmessage::push("E' occorso un errore nel salvataggio del software. <br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);
			return false;
		}

		// Scrive sul log:
		Neikos_Log::Write(sprintf("Nuovo software: (%s)", $id));

		// Scrive nello stack dei messaggi:
		Neikos_Systemmessage::push("Il software &egrave; stato salvato con successo.");

		return $id;
	}

	public function update(array $data, $where)
	{
		try {
			$oldDati = $this->fetchAll($where);

			if (count($oldDati) > 0) {
				$stringa = array();
				foreach ($oldDati as $oldData) {
					$stringa[] = sprintf("Modifica software: (%s), %s, %s, %s, %s, %s, %s, %s", $oldData->id, $oldData->user, base64_encode($oldData->pw), $oldData->project_id, $oldData->user_tipo, $oldData->url, $oldData->url_tipo, base64_encode($oldData->note));
				}
			}
		} catch (Exception $ex) {
			// Nessuna eccezione mostrata...
		}

		try {
			// Setta la data di ultima modifica:
			$data["data_mod"] = date("Y-m-d H:i:s");

			$numRows = parent::update($data, $where);
		} catch(Exception $ex) {
			// Salva un msg di errore nello stack dei msg:
			Neikos_Systemmessage::push("Errore nella modifica dei record.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);

			return false;
		}

		// Scrive sul file di log:
		Neikos_Log::Write($stringa);

		// Scrive nello stack dei messaggi:
		Neikos_Systemmessage::push($numRows . " software " . ($numRows == 1 ? "modificato" : "modificati") . " con successo.");

		return $numRows;
	}

	public function delete($where)
	{
		try {
			$oldDati = $this->fetchAll($where);

			if (count($oldDati) > 0) {
				$stringa = array();
				foreach ($oldDati as $oldData) {
					$stringa[] = sprintf("Eliminazione software: (%s), %s, %s, %s, %s, %s, %s, %s", $oldData->id, $oldData->user, base64_encode($oldData->pw), $oldData->project_id, $oldData->user_tipo, $oldData->url, $oldData->url_tipo, base64_encode($oldData->note));
				}
			}
		} catch (Exception $ex) {
			// Nessuna eccezione mostrata...
		}

		try {
			$numRows = parent::delete($where);
		} catch(Exception $ex) {
			// Salva un msg di errore nello stack dei msg:
			Neikos_Systemmessage::push("Errore nell'eliminazione dei record.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);

			return false;
		}

		// Scrive nel log:
		Neikos_Log::Write($stringa);

		// Salva un msg di OK nello stack dei msg positivi:
		Neikos_Systemmessage::push(sprintf("L'eliminazione &egrave; stata efettuata con successo. Record eliminati: %d", $numRows));

		return $numRows;
	}
}
