<?php

class Software_Plugin extends Zend_Db_Table_Abstract {
	protected $_name = 'software_plugin';
	protected $_dependentTables = array();

	protected $_referenceMap    = array(
	    'Software_Tipo' => array(
	        'columns'           => 'software_tipo_id',
	        'refTableClass'     => 'Software_Tipo',
	        'refColumns'        => 'id'
	    )
	);

	const STATO_ATTIVO = "A";
	const STATO_NON_ATTIVO = "D";

	/**
	 * Lista di plugin di un software formattati in un'array
	 * chiave => valore.
	 *
	 * @param int $softwareTipoId
	 *
	 * @return array
	 */
	public static function getListaForSelect($softwareTipoId = null) {
	    $model = new self();

	    // Query:
	    $sel = $model->select();
	    if (!empty($softwareTipoId)) {
	       $sel->where("software_tipo_id = ?", $softwareTipoId);
	    }
	    $sel->order("name ASC");

	    $rowset = $model->fetchAll($sel);
	    $ret = array();

	    if (count($rowset) > 0) {
	       foreach ($rowset as $row) {
	           if (!empty($softwareTipoId)) {
	               $ret[$row->id] = $row->name;
	           } else {
	               $nomeTipo = Software_Tipo::getNome($row->software_tipo_id);
	               $ret[$nomeTipo][$row->id] = $row->name;
	           }
	       }
	    }

	    return $ret;
	}


	// Filtri di ricerca:
	protected $_filtriRicerca = null;

	/**
	 * Setta i filtri per la lista dei domini.
	 */
	public function setFiltri($filtriRicerca) {
	    $f = array();

	    if (!empty($filtriRicerca->name)) {
	        $f[] =  $this->_db->quoteInto("p.name LIKE ?", "%" . $filtriRicerca->name . "%") .
        	        " OR " .
        	        $this->_db->quoteInto("p.note LIKE ?", "%" . $filtriRicerca->name . "%");
	    }

	    if ($filtriRicerca->software_tipo_id > 0) {
	        $f[] = $this->_db->quoteInto("p.software_tipo_id = ?", $filtriRicerca->software_tipo_id);
	    }

	    $this->_filtriRicerca = $f;
	}


	/**
	 * Prende tutti i dati che saranno visualizzati
	 * in una pagina.
	 *
	 * @param int $numPage - pagina corrente (default: 1).
	 * @param int $maxPerPage - numero massimo di record per pagina (default = 25).
	 * @return mixed
	 */
	public function getPage($numPage = 1, $maxPerPage = 25) {
	    $adapter = $this->getAdapter();

	    $softwareTipo = new Software_Tipo();
	    $cInfo = $softwareTipo->info();
	    $cName = $cInfo[Zend_Db_Table_Abstract::NAME];

	    $select = $this->select()
                	    ->from(array("p" => $this->_name))
                	    ->setIntegrityCheck(false)
                	    ->joinLeft(array("q" => $cName), "p.software_tipo_id = q.ID", array())
                	    ->order(array("q.tipo ASC", "p.name ASC"));

	    if ($numPage >= 0 && $maxPerPage !== null) {
	        $select->limitPage($numPage, $maxPerPage);
	    }

	    // Applica i filtri di ricerca:
	    if (count($this->_filtriRicerca) > 0) {
	        foreach ($this->_filtriRicerca as $f) {
	            $select->where($f);
	        }
	    }

	    return $this->fetchAll($select);
	}

	/**
	 * Conta in numero di record.
	 *
	 * @return int
	 */
	public function count() {
	    $adapter = $this->getAdapter();

	    $softwareTipo = new Software_Tipo();
	    $cInfo = $softwareTipo->info();
	    $cName = $cInfo[Zend_Db_Table_Abstract::NAME];

	    $select = $this->select()
                	    ->from(array("p" => $this->_name), array("num" => "COUNT(p.id)"))
                	    ->setIntegrityCheck(false)
                	    ->joinLeft(array("q" => $cName), "p.software_tipo_id = q.ID", array())
                	    ->order(array("q.tipo ASC", "p.name ASC"));

	    // Applica i filtri di ricerca:
	    if (count($this->_filtriRicerca) > 0) {
	        foreach ($this->_filtriRicerca as $f) {
	            $select->where($f);
	        }
	    }

	    $rows = $this->fetchAll($select);

	    if (count($rows)>0) {
	        return (int)$rows->current()->num;
	    }
	    return 0;
	}


	public function insert(array $data)	{
	    try {
	        // Setta le date di inserimento e modifica:
	        $data["data_ins"] = $data["data_mod"] = date("Y-m-d H:i:s");

	        $id = parent::insert($data);
	    } catch (Exception $ex) {
	        // Segna l'errore nello stack dei msg di errore:
	        Neikos_Systemmessage::push("E' occorso un errore nel salvataggio del plugin software. <br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);
	        return false;
	    }

	    // Scrive sul log:
	    Neikos_Log::Write(sprintf("Nuovo plugin software: (%s)", $id));

	    // Scrive nello stack dei messaggi:
	    Neikos_Systemmessage::push("Il plugin &egrave; stato salvato con successo.");

	    return $id;
	}


	public function update(array $data, $where)
	{
	    try {
	        $oldDati = $this->fetchAll($where);

	        if (count($oldDati) > 0) {
	            $stringa = array();
	            foreach ($oldDati as $oldData) {
	                $stringa[] = sprintf("Modifica plugin software: (%s), %s, %s, %s", $oldData->id, $oldData->software_tipo_id, $oldData->name, base64_encode($oldData->note));
	            }
	        }
	    } catch (Exception $ex) {
	        // Nessuna eccezione mostrata...
	    }

	    try {
	        // Setta la data di ultima modifica:
	        $data["data_mod"] = date("Y-m-d H:i:s");

	        $numRows = parent::update($data, $where);
	    } catch(Exception $ex) {
	        // Salva un msg di errore nello stack dei msg:
	        Neikos_Systemmessage::push("Errore nella modifica dei record.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);

	        return false;
	    }

	    // Scrive sul file di log:
	    Neikos_Log::Write($stringa);

	    // Scrive nello stack dei messaggi:
	    Neikos_Systemmessage::push($numRows . " plugin software " . ($numRows == 1 ? "modificato" : "modificati") . " con successo.");

	    return $numRows;
	}

	public function delete($where)
	{
	    try {
	        $oldDati = $this->fetchAll($where);

	        if (count($oldDati) > 0) {
	            $stringa = array();
	            foreach ($oldDati as $oldData) {
	                $stringa[] = sprintf("Eliminazione plugin software: (%s), %s, %s, %s", $oldData->id, $oldData->software_tipo_id, $oldData->name, base64_encode($oldData->note));
	            }
	        }
	    } catch (Exception $ex) {
	        // Nessuna eccezione mostrata...
	    }

	    try {
	        $numRows = parent::delete($where);
	    } catch(Exception $ex) {
	        // Salva un msg di errore nello stack dei msg:
	        Neikos_Systemmessage::push("Errore nell'eliminazione dei record.<br />" . $ex->__toString(), Neikos_Systemmessage::MSG_ERRORE);

	        return false;
	    }

	    // Scrive nel log:
	    Neikos_Log::Write($stringa);

	    // Salva un msg di OK nello stack dei msg positivi:
	    Neikos_Systemmessage::push(sprintf("L'eliminazione &egrave; stata efettuata con successo. Record eliminati: %d", $numRows));

	    return $numRows;
	}
}
